/***************************************************************
 *                   Materials Object Library                  *
 *Class cubic_tensor : declaration for cubic symmetry materials*
 *                    simula.plus@cemes.fr                     *
 *                   GNU/linux version 3.2.0                   *
 *            software under General Public License            *
 ***************************************************************
 * copyright © 2006,2008,2012 COLLARD Christophe
 * copyright © 2006,2008,2012 Centre National de la Recherche Scientifique
 * copyright © 2006,2008 Arts et Métiers ParisTech
 * copyright © 2006 Université de Valenciennes et du Hainaut-Cambrésis
 * copyright © 2006,2008 Laboratoire de Physique et Mécanique des Matériaux (LPMM - CNRS)
 * copyright © 2006 Laboratoire de Mathématiques et ses Applications de Valenciennes (LAMAV)
 * copyright © 2012 Centre d'Elaboration de Matériaux et d'Etudes Structurales (CEMES - CNRS)
 ***************************************************************/

/*! \namespace materiol
    \brief Materials Object Libraries
*/

/*! \class materiol::cubic_tensor
    \brief \f$ 4^{th} \f$ order cubic tensor computation for materials with cubic symmetry

    \htmlonly 
    <FONT color="#838383">

    cubic tensor belongs to Materials Object Libraries (MateriOL++) </br>
    MateriOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    We define a cubic tensor \f$ T \f$ through three constants : \f$ c_{11}, c_{44}, c_{12} \f$. Its non zero components read \n
    \f$ \displaystyle T_{1111} = T_{2222} = T_{3333} = c_{11} \f$ \n
    \f$ T_{1212} = T_{1221} = T_{2112} = T_{2121} = T_{1313} = T_{1331} = c_{44} \f$ \n
    \f$ T_{3113} = T_{3131} = T_{2323} = T_{2332} = T_{3223} = T_{3232} = c_{44} \f$ \n
    \f$ \displaystyle T_{1122} = T_{2211} = T_{1133} = T_{3311} = T_{2233} = T_{3322} = c_{12} \f$ \n

    The associated reduced symmetric matrix reads (see detailed description of cubic_elasticity_tensor) : \n
     \f$  \begin{pmatrix} c_{11} & c_{12} & c_{12} & 0      & 0      & 0 \\ 
                                 & c_{11} & c_{12} & 0      & 0      & 0 \\
			         &        & c_{11} & 0      & 0      & 0 \\
			         & sym    &        & c_{44} & 0      & 0 \\
			         &        &        &        & c_{44} & 0 \\
			         &        &        &        &        & c_{44} \\
        \end{pmatrix} 
	\f$ \n
	and we get the following symmetries : \f$T_{ijkl} = T_{jikl} = T_{ijlk} = T_{klij} \f$.

    \author copyright \htmlonly &#169; \endhtmlonly 2006, 2008, 2012 Christophe COLLARD \n
            copyright \htmlonly &#169; 2006, 2008, 2012 Centre National de la Recherche Scientifique \endhtmlonly \n
	    copyright \htmlonly &#169; 2006, 2008 Arts et M&#233;tiers ParisTech \endhtmlonly \n
	    copyright \htmlonly &#169; 2006 Universit&#233; de Valenciennes et du Hainaut Cambr&#233;sis \endhtmlonly \n
            copyright \htmlonly &#169; 2006, 2008 Laboratoire de Physique et M&#233;canique des Mat&#233;riaux (LPMM - CNRS) \endhtmlonly \n
	    copyright \htmlonly &#169; 2006 Laboratoire de Math&#233;matiques et ses Applications de Valenciennes (LAMAV) \endhtmlonly \n
	    copyright \htmlonly &#169; 2012 Centre d'Elaboration de Mat&#233;riaux et d'Etudes Structurales (CEMES - CNRS) \endhtmlonly \n
    \version 3.2.0
    \date 2006-2012
    \bug none
    \warning none
*/

#ifndef __cplusplus
#error Must use C++ for the type cubic tensor
#endif

#ifndef __cubic_tensors_hpp
#define __cubic_tensors_hpp


#ifndef __iostream
#include <iostream>
#endif

#ifndef __fstream
#include <fstream>
#endif

#ifndef __assert_h
#include <assert.h>
#endif

#ifndef __math_h
#include <math.h>
#endif

#ifndef __string_h
#include <string.h>
#endif

#ifndef __vectors_hpp
#include "MOL++/vectors.hpp"
#endif

#ifndef __tensors4_hpp
#include "MOL++/tensors4.hpp"
#endif

using namespace std;
using namespace mol;

namespace materiol
{


//=======================================================
template <class T> class cubic_tensor : public tensor4<T>
//=======================================================
{
  using tensor4<T>::size;
  using tensor4<T>::tsr;

  protected :
    T C11, C44, C12; // material constants
    string tensor_name;

    void build_tensor ();  // computes the 4th order isotropic tensor

  public :
    cubic_tensor (string = "");  // default constructor
    cubic_tensor (T, T, T, string = "");
    cubic_tensor (const vector<T>&, string = "");
    cubic_tensor (const tensor4<T>&, string = "");  // cast convertion
    ~cubic_tensor () {}  // destructor

    string name(string = "");  // returns / computes tensor's name
    vector<T> elastic_constants ();
    cubic_tensor<T> inv ();

    template <class Tf> friend ostream& operator << (ostream&, cubic_tensor<Tf>&);
};


//=====Private methods for cubic tensor=================================================


//------------------------------------------------------
template <class T> void cubic_tensor<T>::build_tensor ()
//------------------------------------------------------
{
  if (!size)  // used for de derivated classes
    { tsr = new tensor3<T> [size=3];
      for (int i=0; i<size; i++)
	tsr[i] =& tensor3<T> (size,size,size);
    }

  (*this)(1,1,1,1) = (*this)(2,2,2,2) = (*this)(3,3,3,3) = C11;
  (*this)(1,2,1,2) = (*this)(1,2,2,1) = (*this)(2,1,1,2) = (*this)(2,1,2,1) = (*this)(1,3,1,3) = (*this)(1,3,3,1) = (*this)(3,1,1,3) = (*this)(3,1,3,1) = (*this)(2,3,2,3) = (*this)(2,3,3,2) = (*this)(3,2,2,3) = (*this)(3,2,3,2) = C44;
  (*this)(1,1,2,2) = (*this)(2,2,1,1) = (*this)(1,1,3,3) = (*this)(3,3,1,1) = (*this)(2,2,3,3) = (*this)(3,3,2,2) = C12;
}


//=====Public methods for cubic tensor==================================================


/*!
  \brief Constructor for \f$ \displaystyle 4^{th} \f$ order cubic tensors.

  \param name tensor name
*/

//------------------------------------------------------------
template <class T> cubic_tensor<T>::cubic_tensor (string name)
//------------------------------------------------------------
{
  C11 = C44 = C12 = 0;
  tensor_name = name;
}


/*!
  \brief Constructor for \f$ \displaystyle 4^{th} \f$ order cubic tensors.

  Computes \f$ C_{ijkl} \f$ (see detailed description).


  \param c11 \f$ c_{11} \f$ value
  \param c44 \f$ c_{44} \f$ value
  \param c12 \f$ c_{12} \f$ value
  \param name tensor name
*/

//-------------------------------------------------------------------------------------------------
template <class T> cubic_tensor<T>::cubic_tensor (T c11, T c44, T c12, string name) : tensor4<T>(3)
//-------------------------------------------------------------------------------------------------
{
  C11 = c11;
  C44 = c44;
  C12 = c12;
  build_tensor();
  tensor_name = name;
}


/*!
  \brief Constructor for \f$ \displaystyle 4^{th} \f$ order cubic tensors.

  Computes \f$ C_{ijkl} \f$ (see detailed description).


  \param v \f$ (c_{11} , c_{44} , c_{12}) \f$ values
  \param name tensor name
*/

//------------------------------------------------------------------------------------------------
template <class T> cubic_tensor<T>::cubic_tensor (const vector<T>& v, string name) : tensor4<T>(3)
//------------------------------------------------------------------------------------------------
{
  assert (v.dim() == 3);

  C11 = v[1];
  C44 = v[2];
  C12 = v[3];
  build_tensor();
  tensor_name = name;
}


/*!
  \brief Constructor for cast conversion.

  Converts a \f$ \displaystyle 4^{th} \f$ order tensor into a cubic tensor.

  \param t \f$ \displaystyle 4^{th} \f$ order tensor
  \param name tensor name
*/

//---------------------------------------------------------------------------------
template <class T> cubic_tensor<T>::cubic_tensor (const tensor4<T>& t, string name)
//---------------------------------------------------------------------------------
{
  assert(t.dim1()==3 && t.dim2()==3 && t.dim3()==3 && t.dim4()==3);

  C11 = t(1,1,1,1);
  C12 = t(1,1,2,2);
  C44 = t(1,2,1,2);
  tensor_name = name;
  build_tensor();
  assert ((tensor4<T>)(*this) == t);
}


/*!
  \brief Changes and/or returns tensor name.
  \param name tensor name
  \return tensor name
*/

//-----------------------------------------------------------
template <class T> string cubic_tensor<T>::name (string name)
//-----------------------------------------------------------
{
  if (name != "") tensor_name = name;

  return tensor_name;
}


/*!
  \brief Returns elastic constants.
  \return \f$ (c_{11}, c_{44}, c_{12}) \f$
*/

//----------------------------------------------------------------
template <class T> vector<T> cubic_tensor<T>::elastic_constants ()
//----------------------------------------------------------------
{
  vector<T> constants (3,false);
  constants[1] = C11;
  constants[2] = C44;
  constants[3] = C12;

  return constants;
}


/*!
  \brief Computes the inverse of a \f$ \displaystyle 4^{th} \f$ order cubic tensor.

  If we note \f$ S = C^{-1} \f$, the non zero elements of \f$ S \f$ read : \n
  \f$ \displaystyle S_{1111} = S_{2222} = S_{3333} = \frac{C_{11} + C_{12}}{(C_{11}-C_{12})(C_{11} + 2 C_{12})} \f$ \n
  \f$ \displaystyle S_{1122} = S_{2211} = S_{1133} = S_{3311} = S_{2233} = S_{3322} = -\frac{C_{12}}{(C_{11}-C_{12})(C_{11} + 2 C_{12})} \f$ \n
  \f$ \displaystyle S_{1212} = S_{1221} = S_{2112} = S_{2121} = S_{1313} = S_{1331} = \frac{1}{C_{44}} \f$ \n
  \f$ \displaystyle S_{3113} = S_{3131} = S_{2323} = S_{2332} = S_{3223} = S_{3232} = \frac{1}{C_{44}} \f$ \n
  \return inverse tensor
*/

//--------------------------------------------------------
template <class T> cubic_tensor<T> cubic_tensor<T>::inv ()
//--------------------------------------------------------
{
  T val = 1 / ( (C11 - C12) * (C11 + 2*C12));

  return cubic_tensor<T> ( (C11+C12)*val, 0.25/C44, -C12 * val ); // l'encodage (symtensor -> matrix) utilise n'est pas celui de la doc => laisser le 0.25 / C44 !!
}


//--------------------------------------------------------------------------
template <class Tf> ostream& operator << (ostream& s, cubic_tensor<Tf>& tsr)
//--------------------------------------------------------------------------
{
  assert (tsr.C11 * tsr.C44 * tsr.C12);

  s << tsr.tensor_name << endl;
  s << "Material constants : (" << tsr.C11 << ", " << tsr.C44 << ", " << tsr.C12 << ")" << endl;
  s << (tensor4<Tf>) tsr;
  s << endl;

  return s;
}


}


#endif
